/**
 * URM – Control Panel
 * 
 * Handles creation, deletion, and management of user roles and their duration.
 * Integrates shortcodes, cron scheduling, process logging and JSON storage.
 * Intended for use with URM system backend.
 * 
 * Author: Black & Copilot
 */

// 🛠 URM - Control panel

function render_ur_dashboard_tab() {
    ?>

<div style="display: flex; gap: 20px;">
  <!-- Left box: Add user role -->
  <div style="flex: 1; background: #fff; border: 1px solid #ccc; border-radius: 0px; overflow: hidden; box-shadow: 0 1px 3px rgba(0,0,0,0.05);">

    <!-- Frame header -->
    <div style="background: #f7f9fa; padding: 10px 16px; border-bottom: 1px solid #ccc; font-weight: bold; font-size: 16px;">
      🎭 Add user role
    </div>

    <!-- Content of the frame -->
    <div style="padding: 20px;">
      <form method="post" action="">
		<label for="role_name" style="font-weight: bold;">Add role:</label><br>
        <input type="text" name="add_role_name" placeholder="Enter the role name" style="width: 200px; margin-bottom: 10px;">
        <button type="submit" name="add_role" style="padding: 7px 10px; background: #198754; color: white; border: none; border-radius: 4px; cursor: pointer;">✨ Add</button>
        <p style="color: black; font-size: 14px; margin-top: 10px;">
          By adding a user role, you create a new role in WordPress. These roles will appear in the role menus in all plugins used for role manipulation. In those additions, you can determine which options you want to assign to newly created roles. This tool is only used for creating new roles and follows certain rules. When you create a role with multiple words, you must register it as one word in this way, for example: 'User_1_month'. Don't worry, your role will be displayed through a short code and messages in this way: 'User 1 month'.
        </p>
      </form>
    </div>
  </div>

<!-- Right frame: Delete user role -->
<div style="flex: 1; background: #fff; border: 1px solid #ccc; border-radius: 0px; overflow: hidden; box-shadow: 0 1px 3px rgba(0,0,0,0.05);">

  <!-- Frame header -->
  <div style="background: #f7f9fa; padding: 10px 16px; border-bottom: 1px solid #ccc; font-weight: bold; font-size: 16px;">
    🎭 Delete user role
  </div>

    <!-- Content of the frame -->
    <div style="padding: 20px;">
      <form method="post" action="">
		<label for="role_name" style="font-weight: bold;">Delete role:</label><br>
        <input type="text" name="delete_role_name" placeholder="Enter the role name" style="width: 200px; margin-bottom: 10px;">
        <button type="submit" name="delete_role" style="padding: 7px 10px; background: #dc3545; color: white; border: none; border-radius: 4px; cursor: pointer;">🗑️ Delete</button>
        <p style="color: black; font-size: 14px; margin-top: 10px;">
        By entering the role name and clicking the 'Delete' button, the role will be removed from WordPress roles. You will no longer be able to use that role and its shortcodes.  
        All users who use this role will be left without a role. Make sure to transfer users to a new role before deletion, and only then delete the role when you are sure it will not cause any problems.  
        <b>Important!</b> Input is case-sensitive, so if you do not comply with this, the role will not be deleted. You also need to enter the correct role name as one word, e.g.: 'User_1_month'.
        </p>
      </form>
    </div>
  </div>
</div>

<!-- Left box: Add role duration -->
<div style="display: flex; gap: 20px; margin-top: 20px;">

  <div style="flex: 1; background: #fff; border: 1px solid #ccc; border-radius: 0px; overflow: hidden; box-shadow: 0 1px 3px rgba(0,0,0,0.05);">

    <!-- Frame header -->
    <div style="background: #f7f9fa; padding: 10px 16px; border-bottom: 1px solid #ccc; font-weight: bold; font-size: 16px;">
      ⏱️ Add role duration
    </div>

    <!-- Content of the frame -->
    <div style="padding: 20px;">
      <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
        <input type="hidden" name="action" value="add_role_duration">

        <div style="display: flex; flex-wrap: wrap; gap: 20px;">

          <!-- Role name -->
          <div style="flex: 1 1 100px;">
            <label for="role_name" style="font-weight: bold;">Role name:</label><br>
            <input type="text" name="role_name" placeholder="Enter the role name" id="role_name" style="width: 100%;" required>
          </div>

          <!-- Duration of the role -->
          <div style="flex: 1 1 100px;">
            <label for="role_duration" style="font-weight: bold;">Duration of the role:</label><br>
            <select name="role_duration" id="role_duration" style="width: 100%;">
              <option value="10_minutes">Test (10 minutes)</option>
              <option value="1_month">1 Month</option>
              <option value="3_months">3 Months</option>
              <option value="6_months">6 Months</option>
              <option value="1_year">1 Year</option>
            </select>
          </div>

          <!-- Amount of points -->
          <div style="flex: 1 1 100px;">
            <label for="points_amount" style="font-weight: bold;">Amount of points:</label><br>
            <input type="number" name="points_amount" id="points_amount" value="1" min="1" style="width: 100%;" required>
          </div>

          <!-- Type of points -->
          <div style="flex: 1 1 100px;">
            <label for="points_type" style="font-weight: bold;">Type of points:</label><br>
            <select name="points_type" id="points_type" style="width: 100%;" required>
              <option value="first">First</option>
              <option value="second">Second</option>
              <option value="third">Third</option>
            </select>
          </div>

        </div>

        <!-- Button -->
        <div style="margin-top: 25px; text-align: center;">
          <button type="submit" name="add_role_duration" style="padding: 7px 10px; background: #0d6efd; color: white; border: none; border-radius: 4px; cursor: pointer;">🧾 Create shortcode in the table</button>
        </div>
      </form>

      <p style="color: black; font-size: 14px; margin-top: 10px;">
        Enter the name of the newly created role as you created it, for example: 'User_1_month', determine its duration, set how many points are needed to obtain the role and what type of points.  
        By clicking the 'Create shortcode in the table' button, the entered values will be saved to the table. Place the obtained shortcode on the page where you want the 'Change Role' button to appear.  
        The duration of the role 'Test (10 minutes)' is only for testing to see if cron correctly registers hooks.
      </p>
    </div>
</div>

<!-- Right frame: Guide to shortcodes -->
<div style="flex: 1; background: #fff; border: 1px solid #ccc; border-radius: 0px; overflow: hidden; box-shadow: 0 1px 3px rgba(0,0,0,0.05);">

  <!-- Header -->
  <div style="background: #f7f9fa; padding: 10px 16px; border-bottom: 1px solid #ccc; font-weight: bold; font-size: 16px;">
    🖇️ Guide to Shortcodes
  </div>

  <!-- Content -->
  <div style="padding: 20px;">
    <p style="color: black; font-size: 14px;">
      If you want to display the current role that the user is using, set this shortcode <b>[message_role_change]</b> anywhere on the frontend page where you decided to display the 'Change Role' button. This short code will also display the change of role.
    </p>
    <p style="color: black; font-size: 14px;">
      If you want to display messages about errors, warnings, and success, you can display them through a shortcode <b>[process_log]</b> anywhere on your frontend page where you have placed the above shortcode.
    </p>
	  <p style="color: black; font-size: 14px;">If you want to display the 'User List' table on the frontend, create a new page and use the shortcode <b>[ur_user_list]</b> which will be displayed only for Administrators, while others will see the message: '⛔ This page is only available to administrators.'. If you want to protect the page from being displayed in the menu, you can use the If Menu plugin.</p>
  </div>
	  </div>
		</div>

        <!-- Frame for the table -->
<div style="margin-top: 20px; background: #fff; border: 1px solid #ccc; border-radius: 0px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); overflow: hidden;">

  <div style="background: #f7f9fa; padding: 10px 16px; border-bottom: 1px solid #ccc; font-weight: bold; font-size: 16px;">
    📋 Role name, role duration, amount of points, type of points and role shortcode
  </div>

  <div style="padding: 20px;">

    <style>
    .urm-log-table th, .urm-log-table td {
        padding: 6px 6px 6px 12px; /* ⬅️ added 12px of left padding */
        border: 1px solid #ccc;
        font-size: 14px;
        vertical-align: middle;
        word-break: break-word;
    }
    .urm-log-table tr:nth-child(even) {
        background-color: #f9f9f9;
    }
    .urm-log-table tr:nth-child(odd) {
        background-color: #ffffff;
    }
    </style>
	  
    <table class="urm-log-table" style="width: 100%; border-collapse: collapse; margin-top: 0px;">
        <thead>
            <tr>
                <th>🆔 ID</th>
                <th>🎭 Role name</th>
                <th>⏱️ Role duration</th>
				<th>🪙 Amount of points</th>
				<th>🧿 Type of points</th>
                <th>🖇️ Role shortcode</th>
                <th>🗑️ Delete</th>
            </tr>
        </thead>
       <tbody>
    <?php
    $json_file = WP_CONTENT_DIR . '/plugins/user-role-membership/roles.json';
    $roles_data = json_decode(file_get_contents($json_file), true);

    if (!empty($roles_data['roles'])) {
        foreach ($roles_data['roles'] as $role_name => $entries) {
            foreach ($entries as $entry) {
                if (isset($entry['duration'])) {
                   echo "<tr>
    <td>" . esc_html($entry['id']) . "</td>
    <td>" . esc_html($role_name) . "</td>
    <td>" . esc_html($entry['duration']) . "</td>
    <td>" . (!empty($entry['points_amount']) ? esc_html($entry['points_amount']) : '') . "</td>
    <td>" . (!empty($entry['points_type']) ? esc_html(ucfirst($entry['points_type'])) : '') . "</td>
    <td>[shortcode_role role='" . esc_html($role_name) . "' duration='" . esc_html($entry['duration']) . "']</td>
    <td>
        <form method='post' action='" . esc_url(admin_url('admin-post.php')) . "'>
            <input type='hidden' name='action' value='delete_role_entry'>
            <input type='hidden' name='delete_role_name' value='" . esc_attr($role_name) . "'>
            <input type='hidden' name='delete_role_id' value='" . esc_attr($entry['id']) . "'>
            <center><button type='submit' style='padding: 5px 10px; background: #dc3545; color: white; border: none; border-radius: 4px; cursor: pointer;'>🗑️ Delete</button></center>
        </form>
    </td>
</tr>";
                }
            }
        }
    } else {
        echo "<tr><td colspan='7'><center>Role not found.</center></td></tr>";
    }
    ?>
</tbody>
    </table>
		
<!-- Button to reset the table -->
<div style="text-align: right; margin-top: 20px;">
  <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" onsubmit="return confirm('⚠️ Are you sure you want to delete all roles from the table? This action is irreversible!')">
	  <input type="hidden" name="action" value="reset_role_entries">
        <button type="submit" style="padding: 6px 16px; background: #dc3545; color: white; border: none; border-radius: 4px; font-weight: bold; cursor: pointer;">
      🧹 Reset the table
		</button>
  </form>
</div>

<?php
// 👇 View all backups
$backup_dir = WP_CONTENT_DIR . '/plugins/user-role-membership/backups/';
$backup_url = content_url('/plugins/user-role-membership/backups/');
$backups = glob($backup_dir . 'roles_backup_*.json');

if (!empty($backups)) {
    echo '<div style="margin-top: 40px; border-top: 1px solid #ccc; padding-top: 20px;">';
    echo '<h3>🔁 Restore from backup</h3>';
    echo '<p style="font-size: 14px; color: gray;">⚠️ By clicking on "Restore" you change the current <code>roles.json</code> with data from the selected backup.</p>';
    echo '<ul style="list-style: none; padding-left: 0;">';

    foreach ($backups as $file) {
        $filename = basename($file);
        $timestamp = str_replace(['roles_backup_', '.json'], '', $filename);
        $formatted = date('d.m.Y. H:i:s', DateTime::createFromFormat('Ymd_His', $timestamp)->getTimestamp());

        echo '<li style="margin-bottom: 15px; padding: 10px 15px; background: #f9f9f9; border-radius: 8px; display: flex; justify-content: space-between; align-items: center;">
            <span>📁 <strong>' . esc_html($formatted) . '</strong></span>
            <div style="display: flex; gap: 8px;">
                <!-- Restore backup -->
                <form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" onsubmit="return confirm(\'⚠️ Are you sure you want to restore this backup?\')">
                    <input type="hidden" name="action" value="restore_role_backup">
                    <input type="hidden" name="filename" value="' . esc_attr($filename) . '">
                    <button type="submit" style="padding: 5px 10px; background: #198754; color: white; border: none; border-radius: 4px; cursor: pointer;">🔄 Restore</button>
                </form>

                <!-- Download backup -->
                <a href="' . esc_url($backup_url . $filename) . '" download style="padding: 4px 10px; background: #0d6efd; color: white; border-radius: 4px;  text-decoration: none;">⬇️ Download</a>

                <!-- Delete backup -->
                <form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" onsubmit="return confirm(\'⚠️ Are you sure you want to delete this backup?\')">
                    <input type="hidden" name="action" value="delete_role_backup">
                    <input type="hidden" name="filename" value="' . esc_attr($filename) . '">
                    <button type="submit" style="padding: 5px 10px; background: #dc3545; color: white; border: none; border-radius: 4px; cursor: pointer;">🗑️ Delete</button>
                </form>
            </div>
        </li>';
    }

    echo '</ul></div>';
}
?>

    <?php
}

function handle_user_roles() {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') return;

    $protected_roles = ['administrator', 'editor', 'author', 'contributor', 'subscriber'];

    if (isset($_POST['add_role'])) {
        $role_name = sanitize_text_field($_POST['add_role_name']);
        if (!empty($role_name) && !in_array($role_name, $protected_roles)) {
            add_role($role_name, ucfirst($role_name), array('read' => true));
            echo '<div class="notice notice-success is-dismissible"><p>The role has been successfully added.</p></div>';
        } else {
            echo '<div class="notice notice-error is-dismissible"><p>❌ You cannot change the default WordPress roles.</p></div>';
        }
    }

    if (isset($_POST['delete_role'])) {
        $role_name = sanitize_text_field($_POST['delete_role_name']);
        if (!empty($role_name) && !in_array($role_name, $protected_roles)) {
            remove_role($role_name);
            echo '<div class="notice notice-success is-dismissible"><p>The role has been successfully deleted.</p></div>';
        } else {
            echo '<div class="notice notice-error is-dismissible"><p>❌ You cannot delete default WordPress roles.</p></div>';
        }
    }
}

add_action('admin_init', 'handle_user_roles');

function create_json_folder_and_file() {
    $folder_path = WP_CONTENT_DIR . '/plugins/user-role-membership';
    $json_file = $folder_path . '/roles.json';

    // Check and create a folder if it does not exist
    if (!file_exists($folder_path)) {
        wp_mkdir_p($folder_path);
    }

    // Check and create a `.json` file if it does not exist
    if (!file_exists($json_file)) {
        $default_data = [
            'roles' => []
        ];
        file_put_contents($json_file, json_encode($default_data, JSON_PRETTY_PRINT));
    }
}

add_action('admin_init', 'create_json_folder_and_file');

function add_role_duration_to_json() {
    if (!isset($_POST['role_name']) || !isset($_POST['role_duration'])) return;

    $role_name = sanitize_text_field($_POST['role_name']);
    $role_duration = sanitize_text_field($_POST['role_duration']);
	$points_amount = intval($_POST['points_amount'] ?? 1);
    $points_type = sanitize_text_field($_POST['points_type'] ?? '');

    // Check default WordPress roles
    $protected_roles = ['administrator', 'editor', 'author', 'contributor', 'subscriber'];
    if (in_array($role_name, $protected_roles)) {
        return;
    }

    $json_file = WP_CONTENT_DIR . '/plugins/user-role-membership/roles.json';

    // If it doesn't exist, we create a JSON file.
    if (!file_exists($json_file)) {
        file_put_contents($json_file, json_encode(['roles' => []], JSON_PRETTY_PRINT));
    }

    // We are retrieving existing data.
    $roles_data = json_decode(file_get_contents($json_file), true);

    if (!isset($roles_data['roles'][$role_name])) {
        $roles_data['roles'][$role_name] = [];
    }

    // Generating an ID number
    // We are counting the total number of all previous entries.
    $all_entries_count = 0;
    foreach ($roles_data['roles'] as $entries_per_role) {
        $all_entries_count += count($entries_per_role);
    }
    $new_id = $all_entries_count + 1;

    // Generating a short code role
    $role_shortcode = "[shortcode_role role='" . esc_html($role_name) . "' duration='" . esc_html($role_duration) . "']";

    set_process_message("ℹ️ Adding a new entry: Role = $role_name, Duration = $role_duration, ID = $new_id, Shortcode = $role_shortcode", 'info');

    // Adding duration in JSON with full recording
    $roles_data['roles'][$role_name][] = [
        'id' => $new_id,
        'duration' => str_replace('_', ' ', $role_duration),
        'shortcode' => $role_shortcode,
        'points_amount' => $points_amount,
        'points_type' => $points_type
    ];

    file_put_contents($json_file, json_encode($roles_data, JSON_PRETTY_PRINT));

    set_process_message("ℹ️ Successfully saved to roles.json", 'info');

    // Refreshing the page
    wp_redirect(admin_url('admin.php?page=ur-membership'));
    exit;
}

add_action('admin_post_add_role_duration', 'add_role_duration_to_json');

function delete_role_entry_from_json() {
    if (!isset($_POST['delete_role_name']) || !isset($_POST['delete_role_id'])) return;

    $role_name = sanitize_text_field($_POST['delete_role_name']);
    $role_id = intval($_POST['delete_role_id']);

    $json_file = WP_CONTENT_DIR . '/plugins/user-role-membership/roles.json';
    $roles_data = json_decode(file_get_contents($json_file), true);

    if (!isset($roles_data['roles'][$role_name])) {
        return;
    }

    // We filter the records and remove the one that matches the ID.
    $roles_data['roles'][$role_name] = array_values(array_filter($roles_data['roles'][$role_name], function ($entry) use ($role_id) {
        return $entry['id'] !== $role_id;
    }));

    // Update `roles.json`
    file_put_contents($json_file, json_encode($roles_data, JSON_PRETTY_PRINT));
	
	set_process_message("ℹ️ Update roles.json is successful", 'info');

    // Refreshing the page after deletion
    wp_redirect(admin_url('admin.php?page=ur-membership'));
    exit;
}

add_action('admin_post_delete_role_entry', 'delete_role_entry_from_json');

function user_has_active_membership_role($user_id, $default_roles = ['subscriber']) {
    $user = get_userdata($user_id);
    $roles = (array) $user->roles;

    foreach ($roles as $role) {
        if (!in_array($role, $default_roles)) {
            return true; // It has some additional, active role.
        }
    }

    return false; // It only has the default (e.g. subscriber)
}

function display_role_change_message() {
    if (!is_user_logged_in()) {
        $icon = '❌';
        $color = 'red';
        $background = '#f8d7da';
        $text = '⛔ You must be logged in to see your role.';
    } else {
        $user = wp_get_current_user();
        $raw_role = !empty($user->roles) ? $user->roles[0] : '';
        $display_role = $raw_role ? format_role_name($raw_role) : 'No role assigned';
        $text = 'Your current role is: <strong>' . esc_html($display_role) . '</strong>';
        $icon = '✅';
        $color = '#198754';
        $background = '#d1e7dd';
    }

    return '<div style="
        background: ' . $background . ';
        border-left: 5px solid ' . $color . ';
        color: ' . $color . ';
        padding: 10px;
        margin: 20px 0;
        border-radius: 10px;
        font-size: 17px;
    ">' . $icon . ' ' . $text . '</div>';
}

add_shortcode('message_role_change', 'display_role_change_message');

function display_role_shortcode($atts) {
    $atts = shortcode_atts([
        'role' => '',
        'duration' => ''
    ], $atts);

if (!is_user_logged_in()) {
    return '<div style="
        background: #f8d7da;
        border-left: 5px solid red;
        color: red;
        padding: 10px;
        margin: 20px 0;
        border-radius: 10px;
        font-size: 17px;
    ">⛔ You must be logged in to see your role.</div>';
}

    $user = wp_get_current_user();
    $current_role = !empty($user->roles) ? ucfirst($user->roles[0]) : '';

    $json_file = WP_CONTENT_DIR . '/plugins/user-role-membership/roles.json';
    $roles_data = json_decode(file_get_contents($json_file), true);

if (!isset($roles_data['roles'][$atts['role']])) {
    return '<div style="
        background: #f8d7da;
        border-left: 5px solid red;
        color: red;
        padding: 10px;
        margin: 20px 0;
        border-radius: 10px;
        font-size: 17px;
    ">⛔ The role was not found in the database.</div>';
}

    $role_data = array_filter($roles_data['roles'][$atts['role']], function ($entry) use ($atts) {
    return $entry['duration'] === str_replace('_', ' ', $atts['duration']);
});

$entry = reset($role_data); // This must immediately follow the filter operation

if (empty($role_data)) {
    return '<div style="
        background: #f8d7da;
        border-left: 5px solid red;
        color: red;
        padding: 10px;
        margin: 20px 0;
        border-radius: 10px;
        font-size: 17px;
    ">⛔ Duration not found for this role.</div>';
}

    ob_start();
    ?>
    <center>
		<div style="border: 1px; padding: 10px; background: transparent;">
            <strong>Role:</strong> <?php echo esc_html(format_role_name($atts['role'])); ?><br>
            <strong>Duration:</strong> <?php echo esc_html($atts['duration']); ?><br>
	    	<strong>Price:</strong> <?php echo esc_html($entry['points_amount']) . ' ' . ucfirst($entry['points_type']); ?><br>
            <form method="post">
            <input type="hidden" name="selected_role" value="<?php echo esc_attr($atts['role']); ?>">
            <input type="hidden" name="selected_duration" value="<?php echo esc_attr($atts['duration']); ?>">
            <button type="submit" name="change_role" style="margin-top: 10px;">Change role</button>
            </form>
        </div>
    </center>		
    <?php
    return ob_get_clean();
}

add_shortcode('shortcode_role', 'display_role_shortcode');

function process_switch_role() {
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_role']) && is_user_logged_in()) {
        $user_id = get_current_user_id();

        // ⛔ Check: the user already has an active time-limited role
        if (user_has_active_membership_role($user_id)) {
            set_process_message('⛔ You have already selected a role. You must wait to expire before a new activation.', 'warning');
            wp_redirect(wp_get_referer() ?: home_url('/membership'));
            exit;
        }

        // ➕ Input and basic check
        $new_role = sanitize_text_field($_POST['selected_role'] ?? '');
        $duration = sanitize_text_field($_POST['selected_duration'] ?? '');

        // 🎯 Normalization of spaces in underscores (e.g. "3 months" ➝ "3_months")
        $duration = str_replace(' ', '_', $duration);

        // 🛡️ Check for empty values
        if (empty($new_role) || empty($duration)) {
            set_process_message('❌ Error: Missing data for role change.', 'error');
            wp_redirect(wp_get_referer() ?: home_url('/membership'));
            exit;
        }

        // ✅ Allowed duration intervals
        $valid_durations = ['1_month', '3_months', '6_months', '1_year'];
        if (!in_array($duration, $valid_durations, true)) {
            set_process_message('⛔ Invalid role duration. Please try again.', 'error');
            wp_redirect(wp_get_referer() ?: home_url('/membership'));
            exit;
        }

        // 🎭 Assignment of roles
        $user = new WP_User($user_id);
        $user->set_role($new_role);

        set_process_message("ℹ️ The role has been changed to: " . esc_html(format_role_name($new_role)) . ".", 'info');

        // 🧠 Storage of temporal data
        $start = time();
        $end = match ($duration) {
            '1_month'   => strtotime('+1 month', $start),
            '3_months'  => strtotime('+3 months', $start),
            '6_months'  => strtotime('+6 months', $start),
            '1_year'   => strtotime('+1 year', $start)
        };

        save_ur_user_role_history($user_id, $new_role, format_duration_label($duration), $start, $end);

        // ⏱️ Cron for role reset
        schedule_role_reset($user_id, str_replace('_', ' ', $duration)); // because your map uses spaces

        wp_redirect($_SERVER['REQUEST_URI']);
        exit;
    }
}

add_action('init', 'process_switch_role');

function schedule_role_reset($user_id, $duration) {
    $duration_mapping = [
        '10 minutes' => 600,
        '1 month' => 2592000,
        '3 months' => 7776000,
        '6 months' => 15552000,
        '1 year' => 31536000
    ];

    if (!isset($duration_mapping[$duration])) {
        set_process_message("⛔ Invalid duration: $duration");
        return;
    }

    // 🧽 Remove the old cron if it exists
    $existing = wp_get_scheduled_event('reset_user_role', [$user_id]);
    if ($existing) {
        wp_unschedule_event($existing->timestamp, 'reset_user_role', [$user_id]);
        set_process_message("🧹 Removed previous cron for user ID $user_id", 'info');
    }

    $reset_time = time() + $duration_mapping[$duration];
    $label = $duration;

    set_process_message("ℹ️ Planned role reset for user ID $user_id for $label.", 'info');
    wp_schedule_single_event($reset_time, 'reset_user_role', [$user_id]);
}

function reset_user_role($user_id) {
    if (!$user_id || !get_userdata($user_id)) {
        set_process_message("⛔ reset_user_role invited without a valid user (ID: " . ($user_id ?? 'null') . ").", 'error');
        return;
    }

    $user = new WP_User($user_id);
    $default_role = get_option('default_role', 'subscriber');

    // 🧠 Log in start
    set_process_message("♻️ Initiating a role reset for user ID $user_id → returning to: $default_role", 'info');

    // 🚀 Reset
    $user->set_role($default_role);

    // 🧽 Try removing the scheduled cron
    $event = wp_get_scheduled_event('reset_user_role', [$user_id]);
    if ($event) {
        wp_unschedule_event($event->timestamp, 'reset_user_role', [$user_id]);
        set_process_message("🧹 Cron task removed after execution for user ID: $user_id", 'info');
    } else {
        set_process_message("ℹ️ No more active cron after execution for user ID: $user_id", 'info');
    }
}

add_action('reset_user_role', 'reset_user_role', 10, 1);

function set_process_message($message, $type = 'info') {
    if (!is_user_logged_in()) return;

    $user_id = get_current_user_id();
    
    // ➕ Log in globally
    add_to_ur_process_log($message, $type);

    // ➕ Message for frontend user
    update_user_meta($user_id, 'ur_process_message', json_encode([
        'message' => $message,
        'type' => $type
    ], JSON_UNESCAPED_UNICODE));
}

function display_process_log() {
    if (!is_user_logged_in()) return '';

    $user_id = get_current_user_id();
    $raw = get_user_meta($user_id, 'ur_process_message', true);
    if (empty($raw)) return '';

    delete_user_meta($user_id, 'ur_process_message');
    $data = json_decode($raw, true);

    $type = $data['type'] ?? 'info';
    $message = esc_html($data['message'] ?? '');

    // Style according to the type of message
    switch ($type) {
        case 'error':
            $icon = '❌';
            $color = 'red';
            $background = '#f8d7da';
            break;
        case 'warning':
            $icon = '⚠️';
            $color = '#664d03';
            $background = '#fff3cd';
            break;
        default: // info
            $icon = '✅';
            $color = '#0f5132';
            $background = '#d1e7dd';
            break;
    }

    return '<div style="
        background: ' . $background . ';
        border-left: 5px solid ' . $color . ';
        color: ' . $color . ';
        padding: 10px;
        margin: 20px 0;
        border-radius: 10px;
        font-size: 17px;
    ">' . $icon . ' ' . $message . '</div>';
}

add_shortcode('process_log', 'display_process_log');

add_action('admin_post_reset_role_entries', 'handle_reset_role_entries');

function handle_reset_role_entries() {
    if (!current_user_can('manage_options')) {
        wp_die('You do not have permission for this action.');
    }

    $json_file = WP_CONTENT_DIR . '/plugins/user-role-membership/roles.json';

    // Backup before deletion
    if (file_exists($json_file)) {
        $backup_dir = WP_CONTENT_DIR . '/plugins/user-role-membership/backups/';
        if (!file_exists($backup_dir)) {
            mkdir($backup_dir, 0755, true);
        }

        $timestamp = date('Ymd_His');
        $backup_file = $backup_dir . 'roles_backup_' . $timestamp . '.json';

        copy($json_file, $backup_file);
    }

    // Reset JSON
    $empty_data = ['roles' => []];
    file_put_contents($json_file, json_encode($empty_data, JSON_PRETTY_PRINT));

    // Message and redirect
    set_process_message('🧹 All roles have been successfully reset. Backup saved.', 'warning');
    wp_redirect(wp_get_referer());
    exit;
}

add_action('admin_post_delete_role_backup', 'handle_delete_role_backup');

function handle_delete_role_backup() {
    if (!current_user_can('manage_options')) {
        wp_die('❌ You do not have permission for this action.');
    }

    $filename = sanitize_file_name($_POST['filename'] ?? '');
    $file = WP_CONTENT_DIR . '/plugins/user-role-membership/backups/' . $filename;

    if (!file_exists($file)) {
        set_process_message('❌ The backup file does not exist.', 'error');
        wp_redirect(wp_get_referer());
        exit;
    }

    unlink($file);
    set_process_message('🗑️ The backup has been successfully deleted.', 'warning');
    wp_redirect(wp_get_referer());
    exit;
}

function handle_restore_role_backup() {
    $filename = sanitize_file_name($_POST['filename'] ?? '');
    $backup_file = WP_CONTENT_DIR . '/plugins/user-role-membership/backups/' . $filename;
    $target_file = WP_CONTENT_DIR . '/plugins/user-role-membership/roles.json';

    if (file_exists($backup_file)) {
        copy($backup_file, $target_file);
        set_process_message('🔄 The backup has been successfully restored.', 'info');
    }

    wp_redirect(wp_get_referer());
    exit;
}

add_action('admin_post_restore_role_backup', 'handle_restore_role_backup');

function format_role_name($role_slug) {
    return ucfirst(str_replace('_', ' ', $role_slug));
}

function format_duration_label($duration) {
    return match ($duration) {
        '1_month'  => '1 month',
        '3_months' => '3 months',
        '6_months' => '6 months',
        '1_year'  => '1 year',
    };
 }
